############################################################################
##
## Copyright 2014  Ken Campbell
##
##   Licensed under the Apache License, Version 2.0 (the "License");
##   you may not use this file except in compliance with the License.
##   You may obtain a copy of the License at
##
##     http://www.apache.org/licenses/LICENSE-2.0
##
##   Unless required by applicable law or agreed to in writing, software
##   distributed under the License is distributed on an "AS IS" BASIS,
##   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
##   See the License for the specific language governing permissions and
##   limitations under the License.
##
#####################################
## Description :
##      The VHDL procs for generating various VHDL Testbench items.
##      
##------------------------------------------------------------------------------

##################################################
##  generate BMF shell with stimulus access and registers
##     from block selected, will generate an entity 
##     with stimulus access port and registers / memory
##     coded for behavoural functionality.  The user
##     I/O and functionality will be added by the user.
proc gui_gen_vhdl_bfm {} {
    global FullLst mmap_tree_tr vhdl_bfm_en MmapLst

    set fn [$vhdl_bfm_en get]
    if {$fn == ""} {
        set fn "gen_bfm.vhd"
        $vhdl_bfm_en insert 0 $fn
    }

    ##  for now get the selected block
    set sid [$mmap_tree_tr selection]
    
    ##  if is not a block  mix or reg,  continue   <<<<-----   needs completing.
    #set it  [$mmap_tree_tr item $sid -values]
    #if {$it != "\{Mem Map\}"} {
    #    set reply [tk_dialog .dia "Selected Item" \
    #    "A Memory Map item must be selected in the Memory Map Tree." "" 0 Ok]
    #    return
    #}
    set reg_lst {}
    set racc_lst {}
    set mem_lst {}
    set tdef_lst {}
    set sig_lst {}
    set addr_lst {}
    ## comes from longest name in default signal names (14)
    set max_sign_len 14
    set reso [get_reso]
    
    foreach m $MmapLst {
        set sm [split $m]
        set found [string first $sid [lindex $sm 1]]
        if {$found < 0 && [lindex $sm 0] != "mmap"} {
            continue
        }
        
        switch [lindex $sm 0] {
            "mmap" {
                set id [lindex $sm 1]
                set rid [find_index $id]
                set start [lindex $rid 1]
                lappend rid [find_index_end $start]
                set map_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 3]]
                set tspec [lindex $map_desc 1]
                set stspec [split $tspec ","]
                #puts $stspec
                set acc_sz [lindex $stspec 1]
                #puts $map_desc
            }
            "blk" {
                ## get the start address of the block
                set blk_name [lindex [split [lindex $sm 1] "."] end]
                set blk_base [lindex $sm 2]
                set blk_sz [expr {[lindex $sm 3] - [lindex $sm 2] + $reso}]
                set blk_abits [size2bit_width $blk_sz]
                #puts "Block size:  $blk_sz  Block bits:  $blk_abits"
            }
            "mem" {
                #get infor from the memory map line
                set mname [lindex [split [lindex $sm 1] "."] end]
                set smadd [lindex $sm 2]
                set emadd [lindex $sm 3]
                #puts "start: $smadd   End: $emadd"
                ## get the memory size in addressable locations.
                set memsz [expr {($emadd - $smadd + $reso) / $reso}]
                #puts "Mem size:  $memsz"
                ##  if max len
                if {[string length $mname] > $max_sign_len} {
                    set max_sign_len [string lengh $mname]
                }
                lappend mem_lst {$mname $smadd $emadd $memsz}
                ## create the type def for this memory array
                set ary_typ $mname
                append ary_typ "_array"
                set tdef "type $ary_typ   is array\(0 to [expr {$memsz - 1}]\) of std_logic_vector\("
                append tdef "[expr {$acc_sz - 1}] downto 0\)\;"
                #puts $tdef
                lappend tdef_lst $tdef
                ## create the signal
                set tsig $mname
                lappend tsig $ary_typ
                lappend sig_lst $tsig
                ## add to address list
                set taddr $mname
                lappend taddr [int2bin $smadd $blk_abits] [int2bin [expr {$emadd}] $blk_abits] [size2bit_width $memsz]
                lappend addr_lst $taddr
                
            }
            "reg" {
                set id [lindex $sm 1]
                set rid [find_index $id]
                set reg_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 2]]
                set reg_det [lindex $reg_desc 0]
                set racc [lindex [split $reg_det ","] 3]
                lappend racc_lst $racc
                #puts $racc
                set rname [lindex [split [lindex $sm 1] "."] end]
                if {[string length $rname] > $max_sign_len} {
                    set max_sign_len [string lengh $rname]
                }
                set raddr [lindex $sm 2]
                set tsig [string tolower $rname]
                lappend tsig "std_logic_vector\([expr {$acc_sz - 1}] downto 0\)"
                lappend sig_lst $tsig
                set taddr [string tolower $rname]
                lappend taddr [int2bin $raddr $blk_abits]
                lappend addr_lst $taddr
            }
        }
        ## end switch
    }
    ## end foreach
    set word "std_logic_vector\([expr {$acc_sz - 1}] downto 0\)"
    set upper_select ""
    set upper_bits [expr {$acc_sz - $blk_abits}]
    set upper_range [expr {$acc_sz - ($acc_sz - $blk_abits)}]
    for {set i 0} {$i < $upper_bits} {incr i} {
        append upper_select "0"
    }
    
    ## output the BFM code
    set fh [open $fn w]
    
    puts $fh "----------------------------------------------------------"
    puts $fh "--  BFM $blk_name"
    puts $fh "----------------------------------------------------------"
    puts $fh ""
    puts $fh "library IEEE\;"
    puts $fh "  use IEEE.STD_LOGIC_1164.all\;"
    puts $fh "  use IEEE.STD_LOGIC_ARITH.all\;"
    puts $fh "  use std.textio.all\;"
    puts $fh "  use work.tb_pkg.all\;"
    puts $fh ""
    puts $fh "entity $blk_name is"
    puts $fh "  port \("
    puts $fh "    --  stimulus Access Port"
    puts $fh "    STM_IN       :  in  stm_sctl\;"
    puts $fh "    STM_OUT      :  out stm_sack"
    puts $fh "  \)\;"
    puts $fh "end $blk_name\;"
    puts $fh ""
    puts $fh ""
    puts $fh "architecture bhv of $blk_name is"
    puts $fh "  -- type defs"
    foreach t $tdef_lst {
        puts $fh "  $t"
    }
    puts $fh "  -- register signals  driven by REG_access"
    foreach s $sig_lst {
        puts $fh "  signal [pad_str [lindex $s 0] $max_sign_len] : [lindex $s 1]\;"
    }
    puts $fh "  -- driven by STIM_access"
    puts $fh "  signal [pad_str stim_addr $max_sign_len] : $word\;"
    puts $fh "  signal [pad_str stim_write_dat $max_sign_len] : $word\;"
    puts $fh "  signal [pad_str rd_req $max_sign_len] : std_logic\;"
    puts $fh "  signal [pad_str wr_req $max_sign_len] : std_logic\;"
    puts $fh "  -- driven by REG_access"
    puts $fh "  signal [pad_str stim_read_dat $max_sign_len] : $word\;"
    puts $fh "  signal [pad_str req_ack $max_sign_len] : std_logic\;"
    puts $fh "  signal [pad_str ready $max_sign_len] : std_logic\;"
    puts $fh ""
    puts $fh "begin"
    puts $fh ""
    puts $fh "-------------------------------------------------"
    puts $fh "--  STIM Reg Access process"
    puts $fh "REG_access:"
    puts $fh "  process"
    puts $fh "    variable v_temp_int  :  integer\;"
    puts $fh "    variable v_was_m     :  integer\;"
    puts $fh "  begin"
    puts $fh "    v_was_m  :=  0\;"
    puts $fh "    if\(STM_IN.rst_n'event and STM_IN.rst_n = '0'\) then"
    foreach s $sig_lst {
        set rm [string first "logic" [lindex $s 1]]
        if {$rm >= 0} {
            puts $fh "      [pad_str [lindex $s 0] $max_sign_len] <= \(others => '0'\)\;"
        } else {
            puts $fh "      [pad_str [lindex $s 0] $max_sign_len] <= \(others => \(others => '0'\)\)\;"
        }
    }
    puts $fh ""
    puts $fh "    -- if is a write access"
    puts $fh "    elsif(wr_req' event and wr_req = '1') then"
    puts $fh "      case stim_addr\([expr {$acc_sz - 1}] downto $upper_range\) is"
    puts $fh "        when \"$upper_select\" =>"
    puts $fh "          case stim_addr\([expr {$upper_range - 1}] downto 0\) is"
    set ridx 0
    #puts $racc_lst
    foreach s $addr_lst {
        set this_acc [lindex $racc_lst $ridx]
        incr ridx
        if {$this_acc == "RO"} {
            #puts $s
            continue
        }
        set ll [llength $s]
        if {$ll <= 2} {
            #puts $s
            puts $fh "            when \"[lindex $s 1]\" =>"
            puts $fh "              [pad_str [lindex $s 0] $max_sign_len] <= stim_write_dat\;"
        }
    }
    puts $fh "            when others =>"
    foreach s $addr_lst {
        set ll [llength $s]
        if {$ll > 2} {
            set mupper [expr {[lindex $s 3] - 1}]
            puts $fh "              if\(stim_addr\([expr {$upper_range - 1}] downto 0\) >= \"[lindex $s 1]\" and"
            puts $fh "                 stim_addr\([expr {$upper_range - 1}] downto 0\) <= \"[lindex $s 2]\"\) then"
            set end_brange [expr {int($reso / 2)}]
            set beg_brange [expr {int($mupper + $end_brange)}]
            puts $fh "                v_temp_int := conv_integer\(unsigned\(stim_addr\($beg_brange downto $end_brange\)\)\)\;"
            set nam [lindex $s 0]
            append nam "\(v_temp_int\)"
            puts $fh "                [pad_str $nam $max_sign_len] <= stim_write_dat\;"
            puts $fh "                v_was_m  :=  1\;"
            puts $fh "              end if\;"
        }
    }
    puts $fh "              assert\(v_was_m = 1\)"
    puts $fh "                report \"Out of bounds write attempt in BFM $blk_name, nothing done.\" & LF"
    puts $fh "              severity note\;"
    puts $fh "          end case\;"
    puts $fh "        when others =>"
    puts $fh "          assert\(false\)"
    puts $fh "            report \"Out of bounds write attempt in BFM $blk_name, nothing done.\" & LF"
    puts $fh "          severity note\;"
    puts $fh "      end case\;"
    puts $fh "      -- acknowlage request"
    puts $fh "      req_ack  <=  '1'\;"
    puts $fh "      wait until wr_req'event and wr_req = '0'\;"
    puts $fh "      req_ack  <=  '0'\;"
    puts $fh ""
    puts $fh "    -- if is a read"
    puts $fh "    elsif (rd_req' event and rd_req = '1') then"
    puts $fh "      case stim_addr\([expr {$acc_sz - 1}] downto $upper_range\) is"
    puts $fh "        when \"$upper_select\" =>"
    puts $fh "          case stim_addr\([expr {$upper_range - 1}] downto 0\) is"
    foreach s $addr_lst {
        set ll [llength $s]
        if {$ll <= 2} {
            puts $fh "            when \"[lindex $s 1]\" =>"
            puts $fh "              stim_read_dat  <=  [lindex $s 0]\;"
        } 
    }
    puts $fh "            when others =>"
    puts $fh "              stim_read_dat  <= \(others => 'U'\)\;"
    foreach s $addr_lst {
        set ll [llength $s]
        if {$ll > 2} {
            set mupper [expr {[lindex $s 3] - 1}]
            puts $fh "              if\(stim_addr\([expr {$upper_range - 1}] downto 0\) >= \"[lindex $s 1]\" and"
            puts $fh "                 stim_addr\([expr {$upper_range - 1}] downto 0\) <= \"[lindex $s 2]\"\) then"
            set end_brange [expr {int($reso / 2)}]
            set beg_brange [expr {int($mupper + $end_brange)}]
            puts $fh "                v_temp_int := conv_integer\(unsigned\(stim_addr\($beg_brange downto $end_brange\)\)\)\;"
            set nam [lindex $s 0]
            append nam "\(v_temp_int\)"
            puts $fh "                stim_read_dat  <= $nam\;"
            puts $fh "                v_was_m  :=  1\;"
            puts $fh "              end if\;"
        }
    }
    puts $fh "              assert\(v_was_m = 1\)"
    puts $fh "                report \"Out of bounds read attempt in BFM $blk_name, nothing done.\" & LF"
    puts $fh "              severity note\;"
    puts $fh "          end case\;"
    puts $fh "        when others =>"
    puts $fh "          stim_read_dat  <= \(others => 'U'\)\;"
    puts $fh "          assert\(false\)"
    puts $fh "            report \"Out of bounds read attempt in BFM $blk_name, nothing done.\" & LF"
    puts $fh "          severity note\;"
    puts $fh "      end case\;"
    puts $fh "      -- acknowlage request"
    puts $fh "      req_ack  <=  '1'\;"
    puts $fh "      wait until rd_req'event and rd_req = '0'\;"
    puts $fh "      req_ack  <=  '0'\;"
    puts $fh ""
    puts $fh "    end if\;"
    puts $fh ""
    puts $fh "    wait on rd_req, wr_req, STM_IN.rst_n\;"
    puts $fh "  end process REG_access\;"
    puts $fh ""
    puts $fh "-------------------------------------------------------------------------------"
    puts $fh "-- STIM Access port processes"
    puts $fh "--"
    puts $fh "STIM_access:"
    puts $fh "  process"
    puts $fh "  begin"
    puts $fh "    if(STM_IN.rst_n' event and STM_IN.rst_n = '0') then"
    puts $fh "      STM_OUT   <=   stm_neut\;"
    puts $fh "    -- if read cycle"
    puts $fh "    elsif(STM_IN.req_n' event and STM_IN.req_n  = '0' and STM_IN.rwn = '1') then"
    puts $fh "      stim_addr      <=  STM_IN.addr\;"
    puts $fh "      rd_req         <=  '1'\;"
    puts $fh "      wait until req_ack' event and req_ack = '1'\;"
    puts $fh "      STM_OUT.rdat       <=   stim_read_dat\;"
    puts $fh "      rd_req         <=  '0'\;"
    puts $fh "      wait for 1 ps\;"
    puts $fh "      STM_OUT.ack_n  <=  '0'\;"
    puts $fh "      wait until STM_IN.req_n' event and STM_IN.req_n = '1'\;"
    puts $fh "      wait for 1 ps\;"
    puts $fh "      STM_OUT  <=  stm_neut\;"
    puts $fh ""
    puts $fh "    -- if Write"
    puts $fh "    elsif(STM_IN.req_n' event and STM_IN.req_n  = '0' and STM_IN.rwn = '0') then"
    puts $fh "      stim_addr      <=  STM_IN.addr\;"
    puts $fh "      stim_write_dat <=  STM_IN.wdat\;"
    puts $fh "      wr_req         <=  '1'\;"
    puts $fh "      wait until req_ack' event and req_ack = '1'\;"
    puts $fh "      wait for 1 ps\;"
    puts $fh "      wr_req         <=  '0'\;"
    puts $fh "      wait for 1 ps\;"
    puts $fh "      STM_OUT.ack_n  <=  '0'\;"
    puts $fh "      wait until STM_IN.req_n' event and STM_IN.req_n = '1'\;"
    puts $fh "      wait for 1 ps\;"
    puts $fh "      STM_OUT  <=  stm_neut\;"
    puts $fh "    end if\;"
    puts $fh ""
    puts $fh "    STM_OUT.rdy_n   <=  ready\;"
    puts $fh "    wait on STM_IN.req_n, STM_IN.rst_n, ready\;"
    puts $fh "  end process STIM_access\;"
    puts $fh ""
    puts $fh "end bhv\;"
    
    close $fh
}


##  generate the defines  stm file.
##    this includes the register name, address
##     register name, valid bit mask
proc gui_gen_defines {} {
    global FullLst mmap_tree_tr vhdl_def_en MmapLst
    
    set fn [$vhdl_def_en get]
    if {$fn == ""} {
        set fn "gen_reg_vars.stm"
        $vhdl_def_en insert 0 $fn
    }
    
    ##  for now get the selected block
    set sid [$mmap_tree_tr selection]
    
    set rlst {}
    set mlst {}
    set mem_lst {}
    foreach m $MmapLst {
        set sm [split $m]
        set found [string first $sid [lindex $sm 1]]
        #puts "Found:  $found   string: $sm"
        if {$found < 0 && [lindex $sm 0] != "mmap"} {
            continue
        }
        
        switch [lindex $sm 0] {
            "mmap" {
                set id [lindex $sm 1]
                set rid [find_index $id]
                set start [lindex $rid 1]
                lappend rid [find_index_end $start]
                set map_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 3]]
                set tspec [lindex $map_desc 1]
                set stspec [split $tspec ","]
                #puts $stspec
                set acc_sz [lindex $stspec 1]
                #puts $map_desc
            }
            "blk" {
                ## get the start address of the block
                set blk_base [lindex $sm 2]
            }
            "mem" {
                set mname [lindex [split [lindex $sm 1] "."] end]
                set mstart [lindex $sm 2]
                set mraddr [format "x%lx" [expr {$mstart - $blk_base}]]
                set tmem $mname
                lappend tmem $mraddr
                lappend mem_lst $tmem
            }
            "reg" {
                set rname [lindex [split [lindex $sm 1] "."] end]
                set raddr [lindex $sm 2]
                set vbits [lsort [lindex $m 3]]
                #puts $vbits
                set braddr [format "x%lx" [expr {$raddr - $blk_base}]]
                lappend rlst "$rname $braddr"
                set mask ""
                for {set i 0} {$i < $acc_sz} {incr i} {
                    #set sbit 1
                    set sbit [lsearch $vbits $i]
                    if {$sbit >= 0} {
                        set mask "1$mask"
                    } else {
                        set mask "0$mask"
                    }
                }
                set mask [bin2hex $mask]
                lappend mlst "$rname $mask"
                #puts $mask
                #puts $braddr
            }
        }
        ## end switch
    }
    ## end foreach
    
    set fh [open $fn w]
    puts $fh "--  Register Base addresses."
    foreach a $rlst {
        puts $fh "DEFINE_VAR [lindex $a 0] [lindex $a 1]"
    }
    
    puts $fh ""
    puts $fh "--  Register Bit Maskes."
    foreach a $mlst {
        puts $fh "DEFINE_VAR MASK_[lindex $a 0] x[lindex $a 1]"
    }
    
    puts $fh ""
    puts $fh "--  Memory  Base addresses."
    foreach m $mem_lst {
        puts $fh "DEFINE_VAR [lindex $m 0]_BASE [lindex $m 1]"
    }
    close $fh
    
}


##  generate the vhdl RTL
proc gui_gen_vhdl {} {
    global FullLst mmap_tree_tr vhdl_rtl_en MmapLst
    
    set fn [$vhdl_rtl_en get]
    if {$fn == ""} {
        set fn "gen_vhdl.vhd"
        $vhdl_rtl_en insert 0 $fn
    }
    
    set mm_base 0
    set mm_reso "byte"
    set mm_width 0
    set blk_start 0
    set pin_lst {}
    set sign_lst {}
    set fld_assign_lst {}
    set fmax_len 0
    set rmax_len 0
    
    
    ##  for now get the selected block
    set sid [$mmap_tree_tr selection]
    set ssid [split $sid "."]
    set len [llength $ssid]
    if {$len != 3} {
        dbg_msg "The selected item must be a Block."
        return
    }
    
    ## get the info from FullLst
    set reg_lst {}
    set reg_addr_lst {}
    set field_lst {}
    foreach l $MmapLst {
        set sl [split $l]
        set found [string first $sid [lindex $sl 1]]
        if {$found < 0 && [lindex $sl 0] != "mmap"} {
            continue
        }
        #puts "Found is: $found"
        switch [lindex $sl 0] {
            "mmap" {
                #puts $l
                set id [lindex $sl 1]
                set rid [find_index $id]
                set start [lindex $rid 1]
                lappend rid [find_index_end $start]
                set map_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 3]]
                set tspec [lindex $map_desc 1]
                set stspec [split $tspec ","]
                #puts $stspec
                set acc_sz [lindex $stspec 1]
                #puts $map_desc
            }
            "mem" {
                continue
            }
            "blk" {
                set blk_sz [expr {[lindex $sl 3] - [lindex $sl 2]}]
                set blk_abits [size2bit_width $blk_sz]
                #puts $l
                puts "Address bit width: $blk_abits"
                set id [lindex $sl 1]
                set rid [find_index $id]
                set start [lindex $rid 1]
                lappend rid [find_index_end $start]
                set blk_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 3]]
                #puts $blk_desc
            }
            "reg" {
                #puts $l
                set id [lindex $sl 1]
                set rid [find_index $id]
                set start [lindex $rid 1]
                lappend rid [find_index_end $start]
                #puts $rid
                set name [lindex [split [lindex $l 1] "."] end]
                set addr [lindex $l 2]
                set baddr [int2bin $addr $blk_abits]
                set treg [string tolower $name]
                lappend treg $baddr
                #puts "Register $name has  addr: $baddr"
                ##  temp  get the entity name from here, block name
                set ent_name [lindex [split [lindex $l 1] "."] end-1]
                set tsig [string tolower $name]
                #puts $name
                set rlen [string length $name]
                if {$rlen > $rmax_len} {
                    set rmax_len $rlen
                }
                lappend sign_lst $tsig
                set reg_desc [lrange $FullLst [lindex $rid 1] [lindex $rid 2]]
                set reg_det [lindex $reg_desc 0]
                ## get the register access mode
                set racc [lindex [split $reg_det ","] end]
                lappend treg $racc
                lappend reg_lst $treg
                ##  start the drive outputs list source reg name
                set tdrv {}
                if {$racc == "RW"} {
                    set tdrv [string tolower $name]
                }
                ##  go through all fields of the register
                foreach r $reg_desc {
                    set sr [split $r ","]
                    if {[lindex $sr 0] != 11} {
                      continue
                    }
                    #puts $r
                    # get info from records
                    set fname "[string tolower [lindex $sr 1]]"
                    ## skip reserved fields.
                    set reser [string first "reserved" $fname]
                    if {$reser >= 0} {
                        continue
                    }
                    ##   calculate longest field name
                    set tlen [string length $fname]
                    if {$tlen > $fmax_len} {
                        set fmax_len $tlen
                    }
                    ##  start  pin definition tpin
                    set tpin $fname
                    set fst   [lindex $sr 2]
                    set fsz   [lindex $sr 3]
                    set fdir  [lindex $sr 4]
                    set fend [expr {$fst + $fsz - 1}]
                    if {$fdir == "RW"} {
                        lappend tpin "out"
                        set fdrv $fname
                        lappend fdrv $fend $fst
                        lappend tdrv $fdrv
                    } else {
                        lappend tpin "in "
                    }
                    if {$fsz == 1} {
                        lappend tpin "std_logic"
                    } else {
                        set fsz [expr {$fsz - 1}]
                        lappend tpin "std_logic_vector\($fsz downto 0\)"
                    }
                    lappend tpin $fst $fsz
                    lappend pin_lst $tpin
                    if {$fdir == "RO"} {
                        if {$fsz > 1} {
                            set tassign "$tsig\($fend downto $fst\)  <=  [lindex $tpin 0]\;"
                        } else {
                            set tassign "$tsig\($fst\)  <=  [lindex $tpin 0]\;"
                        }
                        lappend fld_assign_lst $tassign
                    }
                    #puts $tpin
                }
                if {$tdrv != {}} {
                    lappend fdrv_lst $tdrv
                }
            }
        }
    }
    set fmax_len [expr {$fmax_len + 3}]
    set rmax_len [expr {$rmax_len + 3}]
    set msz_idx [expr {$acc_sz - 1}]
    set addr_sz [expr {$blk_abits - 1}]
    
    #foreach d $fdrv_lst {
    #    puts $d
    #}
    
    ## open file for write
    set fh [open $fn w]
    ## output entity
    puts $fh "library IEEE\;"
    puts $fh "use IEEE.STD_LOGIC_1164.all\;"
    puts $fh "use IEEE.STD_LOGIC_ARITH.all\;"
    puts $fh ""
    puts $fh "entity $ent_name is"
    puts $fh "port \("
    puts $fh "  [pad_str rst_n $fmax_len]: in  std_logic\;"
    puts $fh "  [pad_str clk  $fmax_len]: in  std_logic\;"
    puts $fh "  [pad_str addr $fmax_len]: in  std_logic_vector\($addr_sz downto 0\)\;"
    puts $fh "  [pad_str r_wn $fmax_len]: in  std_logic\;"
    puts $fh "  [pad_str sel $fmax_len]: in  std_logic\;"
    puts $fh "  [pad_str d_in $fmax_len]: in  std_logic_vector\($msz_idx downto 0\)\;"
    puts $fh "  [pad_str d_out $fmax_len]: out std_logic_vector\($msz_idx downto 0\)\;"
    set len [llength $pin_lst]
    set idx 1
    foreach p $pin_lst {
        set tp "  [pad_str [lindex $p 0] $fmax_len]"
        append tp ""
        if {$idx < $len} {
            append tp ": [lindex $p 1] [lindex $p 2]\;"
        } else {
            append tp ": [lindex $p 1] [lindex $p 2]"
        }
        incr idx
        puts $fh $tp
    }
    puts $fh "\)\;"
    puts $fh "end $ent_name\;"
    
    
    puts $fh ""
    puts $fh ""
    puts $fh "architecture rtl of $ent_name is"
    puts $fh ""
    foreach s $sign_lst {
        puts $fh "  signal [string tolower [pad_str $s $rmax_len]]: std_logic_vector\($msz_idx downto 0\)\;"
    }
    puts $fh ""
    puts $fh "begin"
    puts $fh ""
    ## field outputs drive always.
    foreach d $fdrv_lst {
        set sreg [lindex $d 0]
        set tflds [lrange $d 1 end]
        foreach f $tflds {
            set s [lindex $f 2]
            set e [lindex $f 1]
            if {$s != $e} {
                puts $fh "  [pad_str [lindex $f 0] $fmax_len] <= $sreg\($e downto $s\)\;"
            } else {
                puts $fh "  [pad_str [lindex $f 0] $fmax_len] <= $sreg\($e\)\;"
            }
        }
        
    }
    puts $fh ""
    puts $fh "field_if:"
    puts $fh "  process\(clk\)"
    puts $fh "  begin"
    puts $fh "    if\(clk'event and clk = '1'\) then"
    puts $fh "      if\(rst_n = '0'\) then"
    ##  reset assume is 0  at start,   need to fix this depending on field init value  <<<<---------
    foreach s $sign_lst {
        puts $fh "        [pad_str $s $rmax_len] <= \(others => '0'\)\;"
    }
    puts $fh "      else"
    ## each input signal/pin is assigned on the rising edge
    foreach f $fld_assign_lst {
        puts $fh "        $f"
    }
    ##  if write and select
    puts $fh "        if\(r_wn = '0' and sel = '1'\) then"
    puts $fh "          case addr\($addr_sz downto 0\) is"
    foreach r $reg_lst {
        if {[lindex $r end] == "RW"} {
            puts $fh "            when \"[lindex $r 1]\" =>"    
            puts $fh "              [lindex $r 0]  <=  d_in\;"    
        }
    }
    puts $fh "            when others =>"
    puts $fh "              null\;"
    puts $fh "          end case\;"
    puts $fh "        end if\;"
    puts $fh "      end if\;"
    puts $fh "    end if\;"
    puts $fh "  end process field_if\;"
    
    puts $fh ""
    puts $fh "read_drive:"
    puts $fh "  process\(sel, r_wn, addr\)"
    puts $fh "  begin"
    puts $fh "    if\(sel = '1'\) then"
    puts $fh "      case addr\($addr_sz downto 0\) is"
    foreach r $reg_lst {
        puts $fh "        when \"[lindex $r 1]\" =>"    
        puts $fh "          d_out  <= [lindex $r 0]\;"    
    }
    puts $fh "        when others =>"
    puts $fh "          null\;"
    puts $fh "      end case\;"
    puts $fh "    end if\;"
    puts $fh "  end process read_drive\;"
    
    puts $fh "end rtl\;"
    
    ##  close the rtl file
    close $fh
}